unit Main;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, Registry, StdCtrls, DB, ExtCtrls;

const
  DBRoot = HKEY_LOCAL_MACHINE;
  ConfigPath = 'SOFTWARE\Borland\Database Engine';
  BLLPath = 'SOFTWARE\Borland\BLW32';
  Config01 = 'CONFIGFILE01';
  DLLPath01 = 'DLLPATH';
  BLAPIPATH = 'BLAPIPATH';


type
  TInitThread = class(TThread)
  protected
    Status, Cfg, DLL: string;
    BrowseOn, EditOn: boolean;
    procedure Execute; Override;
    procedure UpdateStatusPanel;
    procedure EnableBrowseButtons;
    procedure EnableEditControls;
  public
    constructor Create;
  end;

  TUpdateDLLThread = class(TThread)
  protected
    Status: string;
    procedure Execute; Override;
    procedure UpdateDLLPath;
  public
    constructor Create;
  end;

  TMainForm = class(TForm)
    StatusBar1: TStatusBar;
    Panel1: TPanel;
    GroupBox1: TGroupBox;
    CfgPath: TEdit;
    GroupBox2: TGroupBox;
    DLLPath: TEdit;
    Button1: TButton;
    Button2: TButton;
    CfgBrowse: TOpenDialog;
    DLLBrowse: TOpenDialog;
    procedure FormShow(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure Button2Click(Sender: TObject);
    procedure Button1Click(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  MainForm: TMainForm;
  Registry: TRegistry;

implementation

{$R *.DFM}

constructor TInitThread.Create;
begin
  inherited Create(True);
  Priority := tpNormal;
  FreeOnTerminate := True;
  Resume;
end;

constructor TUpdateDLLThread.Create;
begin
  inherited Create(True);
  Priority := tpHighest;
  FreeOnTerminate := True;
  Resume;
end;

procedure TInitThread.Execute;
begin
  try try
    with MainForm.DLLBrowse do
    begin
      FileName := 'Idapi32.dll';
      Filter := 'BDE32 DLLs (*.dll)|*.DLL|All Files (*.*)|*.*';
      Title := 'Find Borland Database Engine 32-Bit Core DLLs';
      Options := [ofHideReadOnly, ofFileMustExist];
    end;
    with MainForm.CFGBrowse do
    begin
      FileName := 'Idpai32.cfg';
      Filter := 'BDE32 Configuration File (*.cfg)|*.CFG|All Files (*.*)|*.*';
      Title := 'Find BDE32 Configuration File';
      Options := [ofHideReadOnly, ofFileMustExist];
    end;
    BrowseOn := True;
  except
    BrowseOn := False;
  end;
  finally
    Synchronize(EnableBrowseButtons);
  end;

  try try
    Registry := TRegistry.Create;
    Registry.RootKey := DBRoot;
    Registry.LazyWrite := False;
    if Registry.OpenKey(ConfigPath, False) = False then
    begin
      EditOn := False;
      raise Exception.Create('Could not open key: ' + ConfigPath);
    end;
    Cfg := Registry.ReadString(Config01);
    DLL := Registry.ReadString(DLLPath01);
    Status := 'Registry opened';
//    EditOn := True;
    EditOn := False;
  except
    Status := 'Open failed. ' + Exception(ExceptObject).Message;
    ShowMessage(Exception(ExceptObject).Message);    
  end;
  finally
    Synchronize(UpdateStatusPanel);
    if BrowseOn = True then
      Synchronize(EnableEditControls);
  end;
end;

procedure TInitThread.EnableBrowseButtons;
begin
  MainForm.Button1.Enabled := BrowseOn;
  MainForm.Button2.Enabled := BrowseOn;
end;

procedure TInitThread.EnableEditControls;
begin
  MainForm.CfgPath.Text := '';
  MainForm.DLLPath.Text := '';
  MainForm.CfgPath.Enabled := EditOn;
  MainForm.DLLPath.Enabled := EditOn;
  MainForm.CfgPath.Text := Cfg;
  MainForm.DLLPath.Text := DLL;
end;

procedure TInitThread.UpdateStatusPanel;
begin
  MainForm.StatusBar1.Panels[0].Text := Status;
end;

procedure TUpdateDLLThread.Execute;
var
  BLLEntries: TStringList;
  W: word;
  BLL: string;

begin
  Status := 'NONE';
  Registry.WriteString(DLLPath01, ExtractFilePath(MainForm.DLLBrowse.FileName));
  Registry.CloseKey;
  if Registry.OpenKey(BLLPath, False) = False then
  begin
    Status := 'Error setting BDE Core DLL path';
    raise Exception.Create('Could not open key: ' + BLLPath);
  end;
  BLLEntries := TStringList.Create;
  try try
    Registry.GetValueNames(BLLEntries);
    for W := 0 to BLLEntries.Count - 1 do
    begin
      if CompareText(BLLEntries.Strings[W], BLAPIPATH) = 0 then
        Registry.WriteString(BLAPIPATH, ExtractFilePath(MainForm.DLLBrowse.FileName))
      else
      begin
        BLL := Registry.ReadString(BLLEntries.Strings[W]);
        BLL := ExtractFileName(BLL);
        Registry.WriteString(BLLEntries.Strings[W],
                    ExtractFilePath(MainForm.DLLBrowse.FileName) + BLL);
      end;
    end;
    Status := 'BDE Core DLL path set'
  except
    Status := 'Error setting BDE Core DLL path';
    ShowMessage(Exception(ExceptObject).Message);
  end;
  finally
    BLLEntries.Free;
    Registry.CloseKey;
    if Registry.OpenKey(ConfigPath, False) = False then
      raise Exception.Create('Could not open key: ' + ConfigPath);
    Synchronize(UpdateDLLPath);
  end;
end;

procedure TUpdateDLLThread.UpdateDLLPath;
begin
  MainForm.DLLPath.Text := Registry.ReadString(DLLPath01);
  MainForm.StatusBar1.Panels[0].Text := Status;
  Screen.Cursor := crDefault;
end;



procedure TMainForm.FormShow(Sender: TObject);
begin
  Registry := nil;
  TInitThread.Create;
end;

procedure TMainForm.FormDestroy(Sender: TObject);
begin
  Registry.CloseKey;
  Registry.Free; 
end;

procedure TMainForm.Button2Click(Sender: TObject);
begin
  DLLBrowse.InitialDir := DLLPath.Text;
  if DLLBrowse.Execute = True then
  begin
    Screen.Cursor := crHourGlass;
    Application.ProcessMessages;
    TUpdateDLLThread.Create;
  end;
end;

procedure TMainForm.Button1Click(Sender: TObject);
begin
  CfgBrowse.InitialDir := ExtractFilePath(CfgPath.Text);
  if CfgBrowse.Execute = True then
  begin
    try
      Registry.WriteString(Config01, CfgBrowse.FileName);
      CfgPath.Text := Registry.ReadString(Config01);
      StatusBar1.Panels[0].Text := 'Configuration file path set';
    except
      ShowMessage(Exception(ExceptObject).Message);
      StatusBar1.Panels[0].Text := 'Error setting configuration file path';
    end;
  end;
end;

end.
